<?php

/*
 * Plugin Name: Woocommerce - Интернет-эквайринг UCS
 * Description: Добавляет в Woocommerce возможность приёма платежей картами Visa, MasterCard, Maestro, JCB и других.
 * Version: 1.0
*/

define('THIS_PLUGIN_DIR', dirname(__FILE__));
load_plugin_textdomain('ucs', 'wp-content/plugins/wooucs/languages', 'wooucs/languages');


add_action('plugins_loaded', 'initWooUcs', 0);

function initWooUcs() {

    if (!class_exists('WC_Payment_Gateway')) {
        return;
    }

    class WC_Gateway_Ucs extends WC_Payment_Gateway {
        
        function __construct() {
            $this->id = 'ucs';
            $this->method_title = 'Интернет-эквайринг UCS';
            $this->method_description = 'Добавляет в Woocommerce возможность приёма платежей картами Visa, MasterCard, Maestro, JCB и других.';

            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title');

            if (version_compare(WOOCOMMERCE_VERSION, '2.0.0', '>=') ) {
                add_action('woocommerce_update_options_payment_gateways_' . $this->id, array(&$this, 'process_admin_options'));
            } else {
                add_action('woocommerce_update_options_payment_gateways', array(&$this, 'process_admin_options'));
            }

            add_action('woocommerce_api_wc_gateway_ucs', array(&$this, 'callback'));
        }

        function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => __('Enable/Disable', 'ucs'),
                    'type' => 'checkbox',
                    'label' => __('Enable UCS payment', 'ucs'),
                    'default' => 'yes'
                ),
                'title' => array(
                    'title' => __('Title', 'ucs'),
                    'type' => 'text',
                    'description' => __('This controls the title which the user sees during checkout.', 'ucs'),
                    'default' => __('Payment via credit cards', 'ucs'),
                    'desc_tip' => true,
                ),
                'shopId' => array(
                    'title' => __('Shop ID', 'ucs'),
                    'type' => 'text',
                ),
                'login' => array(
                    'title' => __('Login', 'ucs'),
                    'type' => 'text',
                ),
                'password' => array(
                    'title' => __('Password', 'ucs'),
                    'type' => 'password',
                ),
                'url' => array(
                    'title' => __('Gateway address', 'ucs'),
                    'type' => 'text',
                ),
            );
        }
    
        function process_payment($order_id) {
            global $woocommerce;

            $order = new WC_Order($order_id);

            $orderAmount = $order->get_total();
            $orderAmount = str_replace(',', '.', $orderAmount);
            $orderAmount = number_format($orderAmount, 2, '.', '');

            $code = md5($order->get_id() . '-' . $this->get_option('shopId') . '-' . $this->method_title);

            $arDelivery = array();
            if ($order->get_shipping_total() > 0) {
                $arDelivery['shipping_cost'] = $order->get_shipping_total();
                $arDelivery['shipping_tax_amount'] = $order->get_shipping_tax();
                $arDelivery['full_shipping_cost'] = $arDelivery['shipping_cost'] + $arDelivery['shipping_tax_amount'];
                $arDelivery['shipping_tax_id'] = $arDelivery['shipping_tax_amount'] / $arDelivery['shipping_cost'] * 100;
            }

            $cartItems = array();
            foreach ($order->get_items() as $item) {
                $cartItem['name'] = $item['name'];
                $cartItem['price'] = $item['total'];
                $cartItem['quantity'] = $item['quantity'];
                $cartItem['tax_amount'] = $order->get_item_tax($item) * $cartItem['quantity'];
                $cartItem['full_price'] = $cartItem['price'] + $cartItem['tax_amount'];
                $cartItem['tax_id'] = $cartItem['tax_amount'] / $cartItem['price'] * 100;
                $cartItems[] = $cartItem;
            }

            $soapClient = $this->newSoap();
            $request = $this->prepareOrder(
                $order->get_id(),
                $orderAmount,
                $order->get_customer_id(),
                $order->get_billing_last_name() . ' ' . $order->get_billing_first_name(),
                $order->get_billing_email(),
                add_query_arg('code', $code, add_query_arg('order_id', $order->get_id(), add_query_arg('wc-api', 'WC_Gateway_ucs', home_url( '/' )))),
                add_query_arg('code', $code, add_query_arg('order_id', $order->get_id(), add_query_arg('wc-api', 'WC_Gateway_ucs', home_url( '/' )))),
                $cartItems, 
                $arDelivery
            );

            try {
                $info = $soapClient->register($request);

                $order->update_status('pending', __('Awaiting UCS payment', 'ucs'));

                return array(
                    'result' => 'success',
                    'redirect' => $info->redirect_url . '?session=' . $info->session
                );
            } catch (SoapFault $fault) {
                wc_add_notice(__('Error has occured. Please, contact support service and provide this error code: ', 'ucs') . $fault->faultstring);
            } catch (Exception $e) {
                wc_add_notice(__('Payment error: ', 'ucs') . $e->getMessage());
            }

            return;
        }


        public function callback() {
            $wcOrder = new WC_Order( (int) $_GET['order_id']);

            $code = md5($wcOrder->id . '-' . $this->get_option('shopId') . '-' . $this->method_title);
            if ($code !== $_GET['code']) {

                wc_add_notice(__('Wrong number number or verification code.', 'ucs'), 'error');

            } else {

                $soapClient = $this->newSoap();
                $objStatus = $this->getOrderStatus($wcOrder->id);

                try {
                    $info = $soapClient->get_status($objStatus);
                    switch ($info->status) {
                        case 'acknowledged':
                        case 'not_acknowledged':
                        case 'authorized':
                            WC()->cart->empty_cart();
                            $wcOrder->add_order_note(__('UCS: payment accepted', 'ucs'));
                            $wcOrder->payment_complete();
                        break;

                        case 'canceled':
                        case 'not_authorized':
                            WC()->cart->empty_cart();
                            $wcOrder->update_status('cancelled', __('Payment canceled', 'ucs'));
                            wc_add_notice(__('Payment canceled', 'ucs'), 'error');
                        break;

                        default:
                            WC()->cart->empty_cart();
                        break;
                    }

                } catch (SoapFault $fault) {
                    if ($fault->faultstring === 'INVALID_ORDER') {
                        WC()->cart->empty_cart();
                        $wcOrder->update_status('cancelled', __('Payment canceled', 'ucs'));
                        wc_add_notice(__('Payment canceled', 'ucs'), 'error');
                    } else {
                        wc_add_notice(__('Error has occured. Please, contact support service and provide this error code: ', 'ucs') . $fault->faultstring, 'error');
                    }
                } catch (Exception $e) {
                    wc_add_notice(__('Payment error: ', 'ucs') . $e->getMessage(), 'error');
                }
            }

            wp_redirect( $this->get_return_url( $wcOrder ) );
            exit;
        }


        function newSoap() {
            require_once(THIS_PLUGIN_DIR . '/orderv2.php');

            return new orderv2(null, array(
                'location' => $this->get_option('url'),
                'uri'      => 'http://www.sirena-travel.ru',
                'login'    => $this->get_option('login'),
                'password' => $this->get_option('password'),
                'trace'    => 0,
                'features' => SOAP_SINGLE_ELEMENT_ARRAYS,
                'connection_timeout' => 12));
        }


        function prepareOrder($orderId, $orderSum, $userId, $userName, $userEmail, $okUrl, $faultUrl, $cartItems, $arDelivery) {
            $request = new register();
            $order = new OrderID(); 
            $order->shop_id = (int) $this->get_option('shopId');
            $order->number = $orderId;
            $cost = new Amount();
            $cost->amount = $orderSum;
            $cost->currency = 'RUB';
            $customer = new CustomerInfo();
            if ($userId > 0) {
                $customer->id = $userId;
            }
            $customer->name = $userName;
            $customer->email = $userEmail;
            $description = new OrderInfo();
            $description->paytype = 'card';

            $arItems = array();

            foreach ($cartItems as $cartItem) {
            
                $itemCost = new Amount();
                $itemCost->amount = $cartItem['full_price']; 
                $itemCost->currency = 'RUB';

                $itemTaxes = array(
                        new SoapVar("<tax type='vat'><percentage>".$cartItem['tax_id']."</percentage><amount><currency>".'RUB'."</currency><amount>".$cartItem['tax_amount']."</amount></amount></tax>", XSD_ANYXML),
                    );

                $item = new OrderItem();
                $item->number = $orderId;
                $item->amount = $itemCost;
                $item->typename = 'goods';
                $item->quantity = $cartItem['quantity'];
                $item->name = $cartItem['name'];
                $item->taxes = new SoapVar($itemTaxes, SOAP_ENC_OBJECT);
                
                $arItems[] = new SoapVar($item, SOAP_ENC_OBJECT, null, null, 'OrderItem');            
            }
            
            if ($arDelivery['full_shipping_cost'] > 0) {
                $deliveryCost = new Amount();
                $deliveryCost->amount = round($arDelivery['full_shipping_cost'], 2);
                $deliveryCost->currency = 'RUB';

                $deliveryTaxes = array(
                            new SoapVar("<tax type='vat'><percentage>".$arDelivery['shipping_tax_id']."</percentage><amount><currency>".'RUB'."</currency><amount>".$arDelivery['shipping_tax_amount']."</amount></amount></tax>", XSD_ANYXML),
                        );
                $shipping = new OrderItem();
                $shipping->number = $orderId;
                $shipping->amount = $deliveryCost;
                $shipping->typename = 'services';
                $shipping->quantity = 1;
                $shipping->name = 'delivery';
                $shipping->taxes = new SoapVar($deliveryTaxes, SOAP_ENC_OBJECT);

                $arItems[] = new SoapVar($shipping, SOAP_ENC_OBJECT, null, null, 'OrderItem');
            }
            
            if (wc_tax_enabled()) {
                $description->sales = new SoapVar($arItems, SOAP_ENC_OBJECT);
            }
            
            $language = new PostEntry();
            $language->name = 'Language';
            $language->value = 'ru';
            $returnUrlOk = new PostEntry();
            $returnUrlOk->name = 'ReturnURLOk';
            $returnUrlOk->value = $okUrl;
            $returnUrlFault = new PostEntry();
            $returnUrlFault->name = 'ReturnURLFault';
            $returnUrlFault->value = $faultUrl;
            $cardtype = new PostEntry();
            $cardtype->name = 'ChoosenCardType';
            $cardtype->value = 'VI';
            $request->order = $order;
            $request->cost = $cost;
            $request->customer = $customer;
            $request->description = $description;

            $postdata = new SoapVar(array(
                                            new SoapVar($language, SOAP_ENC_OBJECT, null, null, 'PostEntry'),
                                            new SoapVar($cardtype, SOAP_ENC_OBJECT, null, null, 'PostEntry'),
                                            new SoapVar($returnUrlOk, SOAP_ENC_OBJECT, null, null, 'PostEntry'),
                                            new SoapVar($returnUrlFault, SOAP_ENC_OBJECT, null, null, 'PostEntry'),
                                    ), SOAP_ENC_OBJECT);

            $request->postdata = $postdata;

            return $request;
        }


        function getOrderStatus($orderId) {
            $status = new get_status();
            $order = new OrderID();
            $order->shop_id = (int) $this->get_option('shopId');
            $order->number = $orderId;
            $status->order = $order;
            return $status;
        }

    }

}

function addWooUcs($methods) {
    $methods[] = 'WC_Gateway_Ucs'; 
    return $methods;
}

function wooucsUpdateStatuses() {
    $gateways = WC()->payment_gateways()->payment_gateways();
    if (!array_key_exists('ucs', $gateways)) {
        return;
    }
    $paymentMethod = $gateways['ucs'];


    $wpQuery = new WP_Query(array(
        'post_type' => 'shop_order',
        'post_status' => 'publish',
        'posts_per_page' => '-1',
        'meta_query' => array(
            array(
                'key' => '_payment_method',
                'value' => 'ucs'
            )
        ),
        'tax_query' => array(
            array(
                'taxonomy' => 'shop_order_status',
                'field' => 'slug',
                'terms' => array('pending')
            )
        )
    ));

    $soapClient = $paymentMethod->newSoap();

    $wpOrders = $wpQuery->posts;
    foreach ($wpOrders as $wpOrder) {
        $wcOrder = new WC_Order();
        $wcOrder->populate($wpOrder);

        $objStatus = $paymentMethod->getOrderStatus($wcOrder->id);
        try {
            $info = $soapClient->get_status($objStatus);
            switch ($info->status) {
                case 'acknowledged':
                case 'not_acknowledged':
                case 'authorized':
                    $wcOrder->add_order_note(__('UCS: payment accepted', 'ucs'));
                    $wcOrder->payment_complete();
                break;

                case 'canceled':
                case 'not_authorized':
                    $wcOrder->update_status('cancelled', __('Payment canceled', 'ucs'));
                break;
            }
            echo 'Статус заказа № ', $wcOrder->id, ': ', $info->status, "\r\n";
        } catch (SoapFault $fault) {
            echo 'Ошибка при проверке статуса заказа № ', $wcOrder->id, ': ', $fault->faultcode , '-' , $fault->faultstring, "\r\n";
            if ($fault->faultstring === 'INVALID_ORDER') {
                $wcOrder->update_status('cancelled', __('Payment canceled', 'ucs'));
            }
        } catch (Exception $e) {
            echo 'Ошибка при проверке статуса заказа № ', $o->id, ': PHP Error', "\r\n";
        }

    }

}

add_filter('woocommerce_payment_gateways', 'addWooUcs');

add_filter('wooucs_cron', 'wooucsUpdateStatuses');